"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerDslRoute = registerDslRoute;
var _configSchema = require("@osd/config-schema");
var _common = require("../../common");
var _metrics = require("../metrics");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

// eslint-disable-next-line @typescript-eslint/no-var-requires
const performance = require('perf_hooks').performance;
function registerDslRoute(router, dataSourceEnabled) {
  router.post({
    path: _common.ServiceEndpoints.GetSearchResults,
    validate: {
      body: _configSchema.schema.any()
    }
  }, async (context, request, response) => {
    const {
      query1,
      dataSourceId1,
      query2,
      dataSourceId2
    } = request.body;
    const actionName = query1 && query2 ? _metrics.METRIC_ACTION.COMPARISON_SEARCH : _metrics.METRIC_ACTION.SINGLE_SEARCH;
    const resBody = {};
    if (query1) {
      const {
        index,
        pipeline,
        size,
        ...rest
      } = query1;
      const params = pipeline !== '' ? {
        index,
        size,
        body: rest,
        search_pipeline: pipeline
      } : {
        index,
        size,
        body: rest
      };
      const start = performance.now();
      try {
        let resp;
        const invalidCharactersPattern = /[\s,:\"*+\/\\|?#><]/;
        if (index !== index.toLowerCase() || index.startsWith('_') || index.startsWith('-') || invalidCharactersPattern.test(index)) {
          resBody.errorMessage1 = {
            statusCode: 400,
            body: 'Invalid Index or missing'
          };
        }
        if (pipeline !== '*' && pipeline !== '_none' && pipeline !== '' && !/^[a-zA-Z0-9_\-*]+(,[a-zA-Z0-9_\-*]+)*$/.test(pipeline)) {
          resBody.errorMessage1 = {
            statusCode: 400,
            body: 'Invalid Pipepline'
          };
        }
        if (dataSourceEnabled && dataSourceId1) {
          const client = context.dataSource.opensearch.legacy.getClient(dataSourceId1);
          resp = await client.callAPI('search', params);
        } else {
          resp = await context.core.opensearch.legacy.client.callAsCurrentUser('search', params);
        }
        const end = performance.now();
        context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, actionName, 200, end - start);
        resBody.result1 = resp;
      } catch (error) {
        var _error$body, _error$body2;
        const end = performance.now();
        context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, actionName, error.statusCode, end - start);
        if (error.statusCode !== 404) console.error(error);

        // Template: Error: {{Error.type}} – {{Error.reason}}
        const errorMessage = `Error: ${(_error$body = error.body) === null || _error$body === void 0 || (_error$body = _error$body.error) === null || _error$body === void 0 ? void 0 : _error$body.type} - ${(_error$body2 = error.body) === null || _error$body2 === void 0 || (_error$body2 = _error$body2.error) === null || _error$body2 === void 0 ? void 0 : _error$body2.reason}`;
        resBody.errorMessage1 = {
          statusCode: error.statusCode || 500,
          body: errorMessage
        };
      }
    }
    if (query2) {
      const {
        index,
        pipeline,
        size,
        ...rest
      } = query2;
      const params = pipeline !== '' ? {
        index,
        size,
        body: rest,
        search_pipeline: pipeline
      } : {
        index,
        size,
        body: rest
      };
      const start = performance.now();
      try {
        let resp;
        const invalidCharactersPattern = /[\s,\"*+\/\\|?#><]/;
        if (index !== index.toLowerCase() || index.startsWith('_') || index.startsWith('-') || invalidCharactersPattern.test(index)) {
          throw new Error('Index invalid or missing.');
        }
        if (pipeline !== '*' && pipeline !== '_none' && pipeline !== '' && !/^[a-zA-Z0-9_\-*]+(,[a-zA-Z0-9_\-*]+)*$/.test(pipeline)) {
          resBody.errorMessage1 = {
            statusCode: 400,
            body: 'Invalid Pipepline'
          };
        }
        if (dataSourceEnabled && dataSourceId2) {
          const client = context.dataSource.opensearch.legacy.getClient(dataSourceId2);
          resp = await client.callAPI('search', params);
        } else {
          resp = await context.core.opensearch.legacy.client.callAsCurrentUser('search', params);
        }
        const end = performance.now();
        context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, actionName, 200, end - start);
        resBody.result2 = resp;
      } catch (error) {
        var _error$body3, _error$body4;
        const end = performance.now();
        if (error.statusCode !== 404) console.error(error);
        context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, actionName, error.statusCode, end - start);

        // Template: Error: {{Error.type}} – {{Error.reason}}
        const errorMessage = `Error: ${(_error$body3 = error.body) === null || _error$body3 === void 0 || (_error$body3 = _error$body3.error) === null || _error$body3 === void 0 ? void 0 : _error$body3.type} - ${(_error$body4 = error.body) === null || _error$body4 === void 0 || (_error$body4 = _error$body4.error) === null || _error$body4 === void 0 ? void 0 : _error$body4.reason}`;
        resBody.errorMessage2 = {
          statusCode: error.statusCode || 500,
          body: errorMessage
        };
      }
    }
    return response.ok({
      body: resBody
    });
  });

  // Get Indices
  router.get({
    path: `${_common.ServiceEndpoints.GetIndexes}/{dataSourceId?}`,
    validate: {
      params: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    const params = {
      format: 'json'
    };
    const start = performance.now();
    try {
      const dataSourceId = request.params.dataSourceId;
      let callApi;
      if (dataSourceEnabled && dataSourceId) {
        callApi = context.dataSource.opensearch.legacy.getClient(dataSourceId).callAPI;
      } else {
        callApi = context.core.opensearch.legacy.client.callAsCurrentUser;
      }
      let resp = await callApi('cat.indices', params);
      const remoteConnections = await callApi('transport.request', {
        method: 'GET',
        path: '/_remote/info'
      });
      if (Object.keys(remoteConnections).length > 0) {
        // fetch remote indices if remote clusters exist
        const remoteClusters = Object.keys(remoteConnections).map(key => `${key}:*`).join(',');
        const resolveResponse = await callApi('transport.request', {
          method: 'GET',
          path: `/_resolve/index/${remoteClusters}`
        });
        let remoteIndices = resolveResponse.indices.map(item => ({
          index: item.name,
          format: 'json'
        }));
        resp = resp.concat(remoteIndices);
      }
      const end = performance.now();
      context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, _metrics.METRIC_ACTION.FETCH_INDEX, 200, end - start);
      return response.ok({
        body: resp
      });
    } catch (error) {
      const end = performance.now();
      context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, _metrics.METRIC_ACTION.FETCH_INDEX, error.statusCode, end - start);
      if (error.statusCode !== 404) console.error(error);
      return response.custom({
        statusCode: error.statusCode || 400,
        body: error.message
      });
    }
  });

  // Get Pipelines
  router.get({
    path: `${_common.ServiceEndpoints.GetPipelines}/{dataSourceId?}`,
    validate: {
      params: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    const params = {
      format: 'json'
    };
    const start = performance.now();
    let resBody = {};
    let resp;
    try {
      const dataSourceId = request.params.dataSourceId;
      if (dataSourceEnabled && dataSourceId) {
        resp = (await context.dataSource.opensearch.getClient(dataSourceId)).transport;
        resp = await resp.request({
          method: 'GET',
          path: `${_common.SEARCH_API}/pipeline`
        });
      } else {
        resp = await context.core.opensearch.client.asCurrentUser.transport.request({
          method: 'GET',
          path: `${_common.SEARCH_API}/pipeline`
        });
      }
      resBody = resp.body;
      const end = performance.now();
      context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, _metrics.METRIC_ACTION.FETCH_PIPELINE, 200, end - start);
      return response.ok({
        body: resBody
      });
    } catch (error) {
      const end = performance.now();
      context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, _metrics.METRIC_ACTION.FETCH_PIPELINE, error.statusCode, end - start);
      if (error.statusCode !== 404) console.error(error);
      return response.customError({
        statusCode: error.statusCode || 400,
        body: error
      });
    }
  });
  router.post({
    path: _common.ServiceEndpoints.GetSingleSearchResults,
    validate: {
      body: _configSchema.schema.any()
    }
  }, async (context, request, response) => {
    const {
      query,
      dataSourceId
    } = request.body;
    const resBody = {};
    const {
      index,
      size,
      search_pipeline,
      ...rest
    } = query;
    const params = {
      index,
      size,
      body: rest
    };
    if (typeof search_pipeline === 'string' && search_pipeline.trim() !== '') {
      params.search_pipeline = search_pipeline;
    }
    try {
      // Execute search
      let resp;
      if (dataSourceEnabled && dataSourceId) {
        const client = context.dataSource.opensearch.legacy.getClient(dataSourceId);
        resp = await client.callAPI('search', params);
      } else {
        resp = await context.core.opensearch.legacy.client.callAsCurrentUser('search', params);
      }
      resBody.result = resp;
    } catch (error) {
      var _error$body5, _error$body6;
      if (error.statusCode !== 404) console.error(error);
      const errorMessage = `Error: ${(_error$body5 = error.body) === null || _error$body5 === void 0 || (_error$body5 = _error$body5.error) === null || _error$body5 === void 0 ? void 0 : _error$body5.type} - ${(_error$body6 = error.body) === null || _error$body6 === void 0 || (_error$body6 = _error$body6.error) === null || _error$body6 === void 0 ? void 0 : _error$body6.reason}`;
      resBody.errorMsg = {
        statusCode: error.statusCode || 500,
        body: errorMessage
      };
    }
    return response.ok({
      body: resBody
    });
  });
  router.get({
    path: `${_common.ServiceEndpoints.GetClusterSettings}/{dataSourceId?}`,
    validate: {
      params: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    const params = {
      include_defaults: true
    };
    const start = performance.now();
    try {
      let callApi;
      if (dataSourceEnabled && request.params.dataSourceId) {
        callApi = context.dataSource.opensearch.legacy.getClient(request.params.dataSourceId).callAPI;
      } else {
        callApi = context.core.opensearch.legacy.client.callAsCurrentUser;
      }
      const resp = await callApi('cluster.getSettings', params);
      const end = performance.now();
      context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, _metrics.METRIC_ACTION.FETCH_CLUSTER_SETTINGS, 200, end - start);
      return response.ok({
        body: resp
      });
    } catch (error) {
      const end = performance.now();
      context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, _metrics.METRIC_ACTION.FETCH_CLUSTER_SETTINGS, error.statusCode, end - start);
      if (error.statusCode !== 404) console.error(error);
      return response.custom({
        statusCode: error.statusCode || 400,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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